/***************************************************************************
 *                          Zanoza Modeler v3.x                            *
 *      This is unpublished propietary source code of Zanoza Software.     *
 *      The copyright notice does not evidence any actual or intended      *
 *                    publication of such source code.                     *
 *                                                                         *
 *                 Copyright (c) 2002-2012 Zanoza Software.                *
 *                                                                         *
 *                        All rights reserved.                             *
 ***************************************************************************
 *--------------------------------------------------------------------------
 * @author:   Oleg M.
 *--------------------------------------------------------------------------
 * @purpose:  ZModeler 3.x Color + Bump + Diffuse + EnvMap shader
 *--------------------------------------------------------------------------
 * @histroy: 29.08.2011. Created
<[[
UserName = "Bump+Diff+Env"
Description = "Color, Diffise, Bump and Env map"
Version  = 1.1
Options  = SOLID BUMP SPECULAR ENVIRONMENT FRESNEL
Samplers = BUMPMAP DETAIL ENVMAP
ManualUV1 = v.vUV01.xy
ManualUV2 = v.vUV01.zw
ManualUV3 = v.vUV23.xy
ManualUV4 = v.vUV23.zw
EnvUV = (float2(vReflect.x, vReflect.y)+1)/2
WorldUV = v.vPosition
AlphaRegister = 29

[BUMPMAP]
Variable = g_BumpMap
Register = 0
Mask = RGB AGB
Scale = 32.R
Level = 30.R
AngularLevel = 30.G

[DETAIL]
Variable = g_Diffuse
Register = 1
Mask = R G B A RGB
Scale = 32.G
Level = 30.B
AngularLevel = 30.A

[ENVMAP]
Variable = g_Refl
Register = 2
Mask = RGB
Scale = 32.B
Level = 31.R
AngularLevel = 31.B

[ALPHABLEND]

[ALPHATEST]

[CUSTOM.1]
Name = Opacity Amplifier
Slot = 31.G
Type = float
Min = 0.0
Max = 1.0
Default = 0.0

[CUSTOM.2]
Name = Fresnel
Slot = 31.A
Type = float
Min = 0.001
Max = 0.3
Default = 0.2


]]>
*/

#pragma pack_matrix( row_major ) 

struct VS_INPUT
{
    float4 vPosition        : POSITION;
    float4 vNormal          : NORMAL;
    float4 vTangent         : TANGENT;
    float4 vColorDif        : COLOR0;
    float4 vUV01            : TEXCOORD0;
    float4 vUV23            : TEXCOORD1;
    float4 vDeformPosition  : TEXCOORD5;
    float4 vDeformNormal    : COLOR2;
    uint4  vBones           : BLENDINDICES;
    float4 vWeights         : BLENDWEIGHT;
};

struct VS_OUTPUT
{
    float4 vPosition  : POSITION;   //transformed position
    float4 vColor     : COLOR0;     //vertex color composition {ambient, diffuse, specular, 1}
    float4 vUV01      : TEXCOORD0;
    float4 vUV23      : TEXCOORD1;
    float4 vUVRefl    : TEXCOORD2;  //environment reflection;
    //precomputed stuff:
    float4 vNormal    : TEXCOORD4;  //world-space normal;
    float4 vViewer    : TEXCOORD5;  //camera-space position;
    float4 vBumpLit   : TEXCOORD6;  //tangent-space light;
    float4 vBumpViewer: TEXCOORD7;  //tangent-space position;
};

//----------------------------------------------------------------------
// globals:
//----------------------------------------------------------------------

#include "zmCommon.inl"

float4      g_alpha          : register(c29);  //[minlevel:testvalue, maxlevel, mapaffect, angular level]
float4      g_textureAmp     : register(c30);  //[bump level, bump angular level, diff level, diff angular level]
float4      g_environment    : register(c31);  //[level, opacity strengthening, angular level, fresnel value]
float4      g_textureScale   : register(c32);  //[bump uv scale, diff uv scale, env uv scale, 0]

// textures:
sampler     g_BumpMap        : register(s[0]);
sampler     g_Diffuse        : register(s[1]);
sampler     g_Refl           : register(s[2]);

#ifdef VERTEX_SHADER
VS_OUTPUT mainVS(in VS_INPUT v)
{
    VS_OUTPUT output = (VS_OUTPUT)0;
    output.vNormal = v.vNormal;
    float4 vTangent = v.vTangent;
    output.vPosition = transformInput(v, output.vNormal, vTangent, output.vViewer);
    output.vColor = any(v.vColorDif.rgb) ? v.vColorDif : 1;

    output.vUV01 = v.vUV01;
    output.vUV23 = v.vUV23;
    //bump-map reflection uses slightly modified spehere-env uv mapping:
    output.vUVRefl.xyz = mul(output.vNormal.xyz, (float3x3)g_mView); // see EnvUV on top;

    // bump-transformed light:
    //-----
    float3x3 worldToTangentSpace;
    worldToTangentSpace[0] = mul(vTangent.xyz, (float3x3)g_mWorld);
    worldToTangentSpace[1] = mul(cross(vTangent.xyz, v.vNormal.xyz)*(-v.vTangent.w), (float3x3)g_mWorld);
    worldToTangentSpace[2] = output.vNormal.xyz;
    output.vBumpLit = normalize(float4(mul(worldToTangentSpace, -g_lightDir.xyz), 1));
    output.vBumpViewer = float4(mul(worldToTangentSpace, output.vViewer.xyz), 1);

    return output;
}
#endif //VERTEX_SHADER

#ifdef PIXEL_SHADER
//------------------------------------------------------------
//  PixelShader: generic Dif+Bump+SpecMap+Refl;
//------------------------------------------------------------
float4 mainPS(in VS_OUTPUT v) : COLOR0
{
    float4 oColor;
    float4 CONST_TEXEL   = float4(1,1,1,1);
    float3 vReflect      = v.vUVRefl.xyz;
    float4 BUMPMAP_TEXEL = tex2D(BUMPMAP_SOURCE, BUMPMAP_UV_SOURCE/g_textureScale.x);
    float4 DETAIL_TEXEL  = tex2D(DETAIL_SOURCE, DETAIL_UV_SOURCE/g_textureScale.y);
    float4 ENVMAP_TEXEL  = CONST_TEXEL;
    float4 VCOLOR_TEXEL  = v.vColor;

    float3 normal = normalize(v.vNormal.xyz);
    float3 bumpNormal = normalize((2*(BUMPMAP_INPUT*BUMPMAP_INPUT_MASK)BUMPMAP_MASK)-1);
    float3 bumpLight = normalize(v.vBumpLit.xyz);
    
    //
    // regular vs bump-mapped normals and light:
    // float4 LightTemp is [non-bump diff, non-bump spec, bump-diff, bump spec]
    float4 specData = 0;
#ifdef SPECULAR_ENABLED
    specData.xy = float2(dot(-g_lightDir.xyz, normal), dot(normal, normalize(v.vViewer.xyz-g_lightDir.xyz)));
    specData.zw = float2(dot(bumpLight, bumpNormal), dot(bumpNormal, normalize(bumpLight+v.vBumpViewer.xyz)));
    float4 LightTemp = float4(
                    saturate(specData.x), 
                    lit(specData.x, specData.y, g_colors[2].a).z,
                    saturate(specData.z),
                    lit(specData.z, specData.w, g_colors[2].a).z); 
    
#else
    float4 LightTemp = float4(
                    saturate(dot(-g_lightDir.xyz, normal)), 
                    0,
                    saturate(dot(bumpLight, bumpNormal)),
                    0); 
#endif //SPECULAR_ENABLED
    // alter diffuse light factor by ambient illumitaion addon:
    LightTemp.x = saturate(LightTemp.x + (g_ambientLight.r + g_ambientLight.g + g_ambientLight.b)/3);
    LightTemp.z = saturate(LightTemp.z + (g_ambientLight.r + g_ambientLight.g + g_ambientLight.b)/3);

    //
    // angular level correction of bumpmap
    oColor.a = lerp(saturate(dot(v.vViewer.xyz, normal)), saturate(dot(v.vBumpViewer.xyz, bumpNormal)), g_textureAmp.x);
    
   
    //
    // angular level on bump:
    if (g_textureAmp.y >= 0)
      specData.x = lerp(1, oColor.a, g_textureAmp.y)*g_textureAmp.x;
    else
      specData.x = lerp(1, 1-oColor.a, abs(g_textureAmp.y))*g_textureAmp.x;

    //
    // environment UV computation are wrong in tangent space (vBumpReflect); 
    // however it's suitable at the moment, since I reduce affection (specData.x)
    // when bump-map normal pointing around (0,0,1)
    //
    float3 vBumpReflect   = reflect(bumpNormal, v.vBumpViewer.xyz);
    float3 vNoBumpReflect = v.vUVRefl.xyz;
    vReflect              = normalize(lerp(vNoBumpReflect, vBumpReflect, specData.x*(1-saturate(bumpNormal.z))));
            
    ENVMAP_TEXEL          = tex2D(ENVMAP_SOURCE, ENVMAP_UV_SOURCE*g_textureScale.z);
    
    float3 fEnvironColor = (ENVMAP_INPUT*ENVMAP_INPUT_MASK)ENVMAP_MASK;
    float3 fDiffColor    = (DETAIL_INPUT)DETAIL_MASK;
     
    //
    // angular level on diffuse:
    if (g_textureAmp.w >= 0)
      specData.y = lerp(1, oColor.a, g_textureAmp.w)*g_textureAmp.z;
    else
      specData.y = lerp(1, 1-oColor.a, abs(g_textureAmp.w))*g_textureAmp.z;
     
    //
    // angular level on env:
    if (g_environment.z >= 0)
      specData.z = lerp(1, oColor.a, g_environment.z)*g_environment.x;
    else
      specData.z = lerp(1, 1-oColor.a, abs(g_environment.z))*g_environment.x;

    // fresnel reflection
    specData.w = (1-oColor.a)*(1-oColor.a);
    specData.w = specData.w*specData.w;
    specData.w = saturate(mad(specData.w*(1-oColor.a), 1-saturate(g_environment.w), g_environment.w));
    
    //regular:
    oColor.rgb =
                // ambient: 
                (g_colors[0].rgb + 0.2*v.vColor.rgb + 0.8*fDiffColor*g_ambientLight.rgb)*g_ambientLight.rgb +
                // diffuse component:
                g_colors[1].rgb*v.vColor.rgb*lerp(1, fDiffColor, specData.y*(DETAIL_INPUT_MASK))*
				max(g_colors[3].a,lerp(LightTemp.x, LightTemp.z, specData.x)) + 
                // specular component:
                g_colors[2].rgb*lerp(LightTemp.y, LightTemp.w, specData.x) +                
                // reflection component:
                lerp(0, fEnvironColor, specData.z)*specData.w + 
                // emissive component:
                g_colors[3].rgb*g_colors[3].a;
                

#ifdef ALPHABLEND_ENABLED
    if (g_alpha.w >= 0)
      oColor.a = lerp(1, oColor.a, g_alpha.w);
    else
      oColor.a = lerp(1, 1-oColor.a, abs(g_alpha.w));
    oColor.a = saturate(g_colors[1].a*lerp(1, ALPHABLEND_INPUT, g_alpha.z)*ALPHABLEND_VERTEXCOLOR*oColor.a*oColor.a + 
                        length(fEnvironColor)*g_environment.y)*g_alpha.y;
#else
    oColor.a = g_alpha.y;
#endif//ALPHABLEND_ENABLED
#ifdef ALPHATEST_ENABLED
    clip(oColor.a - g_alpha.x);
#endif//ALPHATEST_ENABLED

    return oColor;
}
#endif //PIXEL_SHADER

