#include "SoundSystem.h"
#include "bass.h"
#include "DebugLog.h"
#include "CleoInstance.h"
#include "CMenuManager.h"
#include <windows.h>

namespace CLEO
{
	HWND CDECL (*CreateMainWindow)(HINSTANCE hinst);

	HWND CDECL OnCreateMainWindow(HINSTANCE hinst)
	{
		if (HIWORD(BASS_GetVersion()) != BASSVERSION)
			Error("An incorrect version of bass.dll has been loaded");
		TRACE("Creating main window...");
		HWND hwnd = CreateMainWindow(hinst);
		if (!GetThisInstance().soundSystem.Init(hwnd))
			TRACE("SoundSystem::Init() failed. Error code: %d", BASS_ErrorGetCode());
		return hwnd;
	}
	
	CPlaceable *camera;

	void SoundSystem::inject(CodeInjector& inj)
	{
		TRACE("Injecting SoundSystem...");
		GameVersionManager& gvm = GetThisInstance().versionManager;
		CreateMainWindow = gvm.TranslateMemoryAddress(MA_CREATE_MAIN_WINDOW_FUNCTION);
		inj.ReplaceFunction(OnCreateMainWindow, gvm.TranslateMemoryAddress(MA_CALL_CREATE_MAIN_WINDOW));
		camera = gvm.TranslateMemoryAddress(MA_CAMERA);
	}

	void EnumerateBassDevices(int& total, int& enabled, int& default_device)
	{
		BASS_DEVICEINFO info;
		for (default_device = -1, enabled = 0, total = 0; BASS_GetDeviceInfo(total, &info); ++total)
		{
			if (info.flags & BASS_DEVICE_ENABLED) ++enabled;
			if (info.flags & BASS_DEVICE_DEFAULT) default_device = total;
			TRACE("Found sound device %d%s: %s", total, default_device == total ?
				" (default)" : "", info.name);
		}
	}

	bool SoundSystem::Init(HWND hwnd)
	{
		const BASS_3DVECTOR pos(0, 0, 0), vel(0, 0, 0), front(0, -1, 0), top(0, 0, 1);
		int default_device, total_devices, enabled_devices;
		BASS_DEVICEINFO info = { nullptr, nullptr, 0 };
		EnumerateBassDevices(total_devices, enabled_devices, default_device);
		if (forceDevice != -1 && BASS_GetDeviceInfo(forceDevice, &info) &&
			info.flags & BASS_DEVICE_ENABLED) 
			default_device = forceDevice;
		TRACE("On system found %d devices, %d enabled devices, assuming device to use: %d (%s)",
			total_devices, enabled_devices, default_device, BASS_GetDeviceInfo(default_device, &info) ?
				info.name : "Unknown device");
		if (BASS_Init(default_device, 44100, BASS_DEVICE_3D, hwnd, nullptr) &&
			BASS_Set3DFactors(1.0f, 0.3f, 1.0f) &&
			BASS_Set3DPosition(&pos, &vel, &front, &top))
		{
			TRACE("SoundSystem initialized");
			initialized = true;
			this->hwnd = hwnd;
			BASS_Apply3D();
			return true;
		}
		Warning("Could not initialize BASS sound system");
		return false;
	}

	SoundSystem::SoundSystem(void)
		: initialized(false), forceDevice(-1), paused(false)
	{
		// TODO: give to user an ability to force a sound device to use (ini-file or cmd-line?)
	}

	SoundSystem::~SoundSystem(void)
	{
		TRACE("Closing SoundSystem...");
		UnloadAllStreams();
		if (initialized) BASS_Free();
	}

	CAudioStream *SoundSystem::LoadStream(const char *filename, bool in3d)
	{
		CAudioStream *result = in3d ? new C3dAudioStream(filename) : new CAudioStream(filename);
		if (result->OK)
		{
			streams.insert(result);
			return result;
		}
		delete result;
		return nullptr;
	}
	
	void SoundSystem::UnloadStream(CAudioStream *stream)
	{
		if (streams.erase(stream))
			delete stream;
		else
			TRACE("Unloading of stream that is not in list of loaded streams");
	}

	void SoundSystem::UnloadAllStreams()
	{
		std::for_each(streams.begin(), streams.end(),
			[](CAudioStream *stream)
			{
				delete stream;
			});
		streams.clear();
	}

	void SoundSystem::Update()
	{
		if (menuManager->menuActive || IsIconic(hwnd))	// if in menu or window minimized
		{
			if (!paused)
			{
				// entering menu - pause all streams
				paused = true;
				std::for_each(streams.begin(), streams.end(),
					[](CAudioStream *stream)
					{
						if (stream->state == CAudioStream::playing)
							stream->Pause(false);
					});
			}
		}
		else
		{
			if (paused)
			{
				// exiting menu - resume all paused streams
				paused = false;
				std::for_each(streams.begin(), streams.end(),
					[](CAudioStream *stream)
					{
						if (stream->state == CAudioStream::playing)
							stream->Resume();
					});
			}
			// not in menu
			// process camera movements
			BASS_Set3DPosition((BASS_3DVECTOR *)&camera->xyz->matrix.pos, nullptr, 
				(BASS_3DVECTOR *)&camera->xyz->matrix.at, (BASS_3DVECTOR *)&camera->xyz->matrix.top);
			// process all streams
			std::for_each(streams.begin(), streams.end(),
				[](CAudioStream *stream)
				{
					stream->Process();
				});
			// apply above changes
			BASS_Apply3D();
		}
	}

	CAudioStream::CAudioStream() 
		: streamInternal(0), state(no), OK(false)
	{ 
	}

	CAudioStream::CAudioStream(const char *src) : state(no), OK(false)
	{
		if (!(streamInternal = BASS_StreamCreateFile(FALSE, src, 0, 0, 0)) &&
			!(streamInternal = BASS_StreamCreateURL(src, 0, 0, nullptr, nullptr)))
		{
			TRACE("Loading audiostream %s failed. Error code: %d", src, BASS_ErrorGetCode());
		}
		else OK = true;
	}

	CAudioStream::~CAudioStream()
	{
		if (streamInternal) BASS_StreamFree(streamInternal);
	}

	C3dAudioStream::C3dAudioStream(const char *src)
		: CAudioStream::CAudioStream(), link(nullptr)
	{
		unsigned flags = BASS_SAMPLE_3D | BASS_SAMPLE_MONO | BASS_SAMPLE_SOFTWARE;
		if (!(streamInternal = BASS_StreamCreateFile(FALSE, src, 0, 0, flags)) &&
			!(streamInternal = BASS_StreamCreateURL(src, 0, flags, nullptr, nullptr)))
		{
			TRACE("Loading 3d-audiostream %s failed. Error code: %d", src, BASS_ErrorGetCode());
		}
		else OK = true;
	}

	C3dAudioStream::~C3dAudioStream()
	{
		if (streamInternal) BASS_StreamFree(streamInternal);
	}

	void CAudioStream::Play()
	{
		BASS_ChannelPlay(streamInternal, TRUE);
		state = playing;
	}

	void CAudioStream::Pause(bool change_state)
	{
		BASS_ChannelPause(streamInternal);
		if (change_state) state = paused;
	}

	void CAudioStream::Stop()
	{
		BASS_ChannelPause(streamInternal);
		BASS_ChannelSetPosition(streamInternal, 0, BASS_POS_BYTE);
		state = paused;
	}

	void CAudioStream::Resume()
	{
		BASS_ChannelPlay(streamInternal, FALSE);
		state = playing;
	}

	unsigned CAudioStream::GetLength()
	{
		return (unsigned)BASS_ChannelBytes2Seconds(streamInternal, 
			BASS_ChannelGetLength(streamInternal, BASS_POS_BYTE));
	}

	unsigned CAudioStream::GetState()
	{
		if (state == stopped) return -1;
		switch (BASS_ChannelIsActive(streamInternal))
		{
		case BASS_ACTIVE_STOPPED:
		default:
			return -1;
		case BASS_ACTIVE_PLAYING:
		case BASS_ACTIVE_STALLED:
			return 1;
		case BASS_ACTIVE_PAUSED:
			return 2;	
		};
	}

	float CAudioStream::GetVolume()
	{
		float result;
		if (!BASS_ChannelGetAttribute(streamInternal, BASS_ATTRIB_VOL, &result))
			return -1.0f;
		return result;
	}

	void CAudioStream::SetVolume(float val)
	{
		BASS_ChannelSetAttribute(streamInternal,BASS_ATTRIB_VOL, val);
	}
	
	void CAudioStream::Loop(bool enable)
	{
		BASS_ChannelFlags(streamInternal, enable ? BASS_SAMPLE_LOOP : 0, BASS_SAMPLE_LOOP);
	}

	void CAudioStream::Process()
	{
		// no actions required
	}

	void CAudioStream::Set3dPosition(const RwV3d& pos)
	{
		TRACE("Unimplemented CAudioStream::Set3dPosition()");
	}
	
	void CAudioStream::Link(CPlaceable *placable)
	{
		TRACE("Unimplemented CAudioStream::Link()");
	}

	void C3dAudioStream::Set3dPosition(const RwV3d& pos)
	{
		BASS_ChannelSet3DPosition(streamInternal, 
			reinterpret_cast<const BASS_3DVECTOR *>(&pos), nullptr, nullptr);
	}

	void C3dAudioStream::Link(CPlaceable *placable)
	{
		link = placable;
	}
	
	void C3dAudioStream::Process()
	{
		// update playing position of the linked object
		if (link && state == playing) 
			BASS_ChannelSet3DPosition(streamInternal, 
			reinterpret_cast<const BASS_3DVECTOR *>(&link->placement.pos), nullptr, nullptr);
	}
}
